/* mhs: A GObject wrapper for the Mozilla Mhs API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>

#include "mhs-error.h"
#include "mhs-error-private.h"

#include "mhs-login-manager-storage.h"
#include "mhs-login-manager-storage-bindings.h"
#include "mhs-service.h"
#include "mhs-marshal.h"

G_DEFINE_TYPE (MhsLoginManagerStorage,
               mhs_login_manager_storage,
               G_TYPE_OBJECT);

#define MHS_LOGIN_MANAGER_STORAGE_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MHS_TYPE_LOGIN_MANAGER_STORAGE, \
                                MhsLoginManagerStoragePrivate))

struct _MhsLoginManagerStoragePrivate
{
  DBusGProxy *proxy;
};

static void
mhs_login_manager_storage_dispose (GObject *object)
{
  MhsLoginManagerStorage *self = (MhsLoginManagerStorage *) object;
  MhsLoginManagerStoragePrivate *priv = self->priv;

  if (priv->proxy)
    {
      g_object_unref (priv->proxy);
      priv->proxy = NULL;
    }

  G_OBJECT_CLASS (mhs_login_manager_storage_parent_class)->dispose (object);
}

static void
mhs_login_manager_storage_class_init (MhsLoginManagerStorageClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  gobject_class->dispose = mhs_login_manager_storage_dispose;

  g_type_class_add_private (klass, sizeof (MhsLoginManagerStoragePrivate));
}

static void
mhs_login_manager_storage_init (MhsLoginManagerStorage *self)
{
  MhsLoginManagerStoragePrivate *priv;
  DBusGConnection *connection;
  GError *error = NULL;

  priv = self->priv = MHS_LOGIN_MANAGER_STORAGE_GET_PRIVATE (self);

  if ((connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
    }
  else
    {
      priv->proxy = dbus_g_proxy_new_for_name
        (connection,
         MHS_SERVICE_LOGIN_MANAGER_STORAGE,
         MHS_SERVICE_LOGIN_MANAGER_STORAGE_PATH,
         MHS_SERVICE_LOGIN_MANAGER_STORAGE_INTERFACE);

      dbus_g_connection_unref (connection);
    }
}

MhsLoginManagerStorage *
mhs_login_manager_storage_new (void)
{
  MhsLoginManagerStorage *self
    = (MhsLoginManagerStorage *) g_object_new (MHS_TYPE_LOGIN_MANAGER_STORAGE,
                                               NULL);

  return self;
}

static gboolean
mhs_lms_check_proxy (MhsLoginManagerStorage  *self,
                     GError                 **error)
{
  if (self->priv->proxy)
    return TRUE;

  g_set_error (error, MHS_ERROR,
               MHS_ERROR_PROXY,
               "Failed to initialize DBUS proxy");

  return FALSE;
}

gboolean
mhs_lms_add_login (MhsLoginManagerStorage *self,
                   const gchar *hostname,
                   const gchar *form_submit_url,
                   const gchar *http_realm,
                   const gchar *username,
                   const gchar *password,
                   const gchar *username_field,
                   const gchar *password_field,
                   GError **error)
{
  MhsLoginManagerStoragePrivate *priv;
  gboolean ret;
  guint32 null_fields = 0;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_lms_check_proxy (self, error))
    return FALSE;

  if (hostname == NULL)
    {
      null_fields |= 1;
      hostname = "";
    }
  if (form_submit_url == NULL)
    {
      null_fields |= 2;
      form_submit_url = "";
    }
  if (http_realm == NULL)
    {
      null_fields |= 4;
      http_realm = "";
    }
  if (username == NULL)
    {
      null_fields |= 8;
      username = "";
    }
  if (password == NULL)
    {
      null_fields |= 16;
      password = "";
    }
  if (username_field == NULL)
    {
      null_fields |= 32;
      username_field = "";
    }
  if (password_field == NULL)
    {
      null_fields |= 64;
      password_field = "";
    }

  ret = org_moblin_mhs_LoginManagerStorage_add_login (priv->proxy,
                                                      hostname,
                                                      form_submit_url,
                                                      http_realm,
                                                      username,
                                                      password,
                                                      username_field,
                                                      password_field,
                                                      null_fields,
                                                      error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_lms_count_logins (MhsLoginManagerStorage *self,
                      const gchar *hostname,
                      const gchar *action_url,
                      const gchar *http_realm,
                      guint *count_p,
                      GError **error)
{
  MhsLoginManagerStoragePrivate *priv;
  gboolean ret;
  guint32 null_fields = 0;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_lms_check_proxy (self, error))
    return FALSE;

  if (hostname == NULL)
    {
      null_fields |= 1;
      hostname = "";
    }
  if (action_url == NULL)
    {
      null_fields |= 2;
      action_url = "";
    }
  if (http_realm == NULL)
    {
      null_fields |= 4;
      http_realm = "";
    }

  ret = org_moblin_mhs_LoginManagerStorage_count_logins (priv->proxy,
                                                         hostname,
                                                         action_url,
                                                         http_realm,
                                                         null_fields,
                                                         count_p,
                                                         error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

static void
mhs_lms_convert_login_array_to_login_infos (GPtrArray *login_array,
                                            guint *n_logins,
                                            MhsLoginInfo **logins)
{
  guint login_num, string_num;

  *n_logins = login_array->len;
  *logins = g_new (MhsLoginInfo, login_array->len);

  for (login_num = 0; login_num < login_array->len; login_num++)
    {
      GValueArray *value_array = g_ptr_array_index (login_array, login_num);
      MhsLoginInfo *login = (* logins) + login_num;
      GValue *null_fields_value;
      guint null_fields;

      null_fields_value = g_value_array_get_nth (value_array,
                                                 sizeof (MhsLoginInfo)
                                                 / sizeof (gchar *));
      null_fields = g_value_get_uint (null_fields_value);

      for (string_num = 0;
           string_num < sizeof (MhsLoginInfo) / sizeof (gchar *);
           string_num++)
        {
          gchar **string_pos = ((gchar **) login) + string_num;

          if ((null_fields & (1 << string_num)))
            *string_pos = NULL;
          else
            {
              GValue *value = g_value_array_get_nth (value_array, string_num);
              *string_pos = g_strdup (g_value_get_string (value));
            }
        }

      g_value_array_free (value_array);
    }

  g_ptr_array_free (login_array, TRUE);
}

gboolean
mhs_lms_find_logins (MhsLoginManagerStorage *self,
                     const gchar *hostname,
                     const gchar *action_url,
                     const gchar *http_realm,
                     guint *n_logins,
                     MhsLoginInfo **logins,
                     GError **error)
{
  MhsLoginManagerStoragePrivate *priv;
  GPtrArray *login_array;
  guint32 null_fields = 0;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_lms_check_proxy (self, error))
    return FALSE;

  if (hostname == NULL)
    {
      null_fields |= 1;
      hostname = "";
    }
  if (action_url == NULL)
    {
      null_fields |= 2;
      action_url = "";
    }
  if (http_realm == NULL)
    {
      null_fields |= 4;
      http_realm = "";
    }

  if (org_moblin_mhs_LoginManagerStorage_find_logins (priv->proxy,
                                                      hostname,
                                                      action_url,
                                                      http_realm,
                                                      null_fields,
                                                      &login_array,
                                                      error))
    {
      mhs_lms_convert_login_array_to_login_infos (login_array,
                                                  n_logins, logins);

      return TRUE;
    }
  else
    {
      _mhs_error_translate_from_dbus (error);

      return FALSE;
    }
}

gboolean
mhs_lms_get_all_disabled_hosts (MhsLoginManagerStorage *self,
                                gchar ***hostnames_out,
                                GError **error)
{
  MhsLoginManagerStoragePrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_lms_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_LoginManagerStorage_get_all_disabled_hosts
    (priv->proxy, hostnames_out,
     error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_lms_get_all_logins (MhsLoginManagerStorage *self,
                        guint *n_logins,
                        MhsLoginInfo **logins,
                        GError **error)
{
  MhsLoginManagerStoragePrivate *priv;
  GPtrArray *login_array;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_lms_check_proxy (self, error))
    return FALSE;

  if (org_moblin_mhs_LoginManagerStorage_get_all_logins (priv->proxy,
                                                         &login_array,
                                                         error))
    {
      mhs_lms_convert_login_array_to_login_infos (login_array,
                                                  n_logins, logins);

      return TRUE;
    }
  else
    {
      _mhs_error_translate_from_dbus (error);

      return FALSE;
    }
}

gboolean
mhs_lms_get_all_encrypted_logins (MhsLoginManagerStorage *self,
                                  guint *n_logins,
                                  MhsLoginInfo **logins,
                                  GError **error)
{
  MhsLoginManagerStoragePrivate *priv;
  GPtrArray *login_array;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_lms_check_proxy (self, error))
    return FALSE;

  if (org_moblin_mhs_LoginManagerStorage_get_all_encrypted_logins (priv->proxy,
                                                                   &login_array,
                                                                   error))
    {
      mhs_lms_convert_login_array_to_login_infos (login_array,
                                                  n_logins, logins);

      return TRUE;
    }
  else
    {
      _mhs_error_translate_from_dbus (error);

      return FALSE;
    }
}

gboolean
mhs_lms_get_login_saving_enabled (MhsLoginManagerStorage *self,
                                  const gchar *host,
                                  gboolean *enabled_out,
                                  GError **error)
{
  MhsLoginManagerStoragePrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_lms_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_LoginManagerStorage_get_login_saving_enabled
    (priv->proxy, host, enabled_out,
     error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_lms_init (MhsLoginManagerStorage *self,
              GError **error)
{
  MhsLoginManagerStoragePrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_lms_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_LoginManagerStorage_init (priv->proxy,
                                                 error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_lms_init_with_file (MhsLoginManagerStorage *self,
                        const gchar *input_file,
                        const gchar *output_file,
                        GError **error)
{
  MhsLoginManagerStoragePrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_lms_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_LoginManagerStorage_init_with_file (priv->proxy,
                                                           input_file,
                                                           output_file,
                                                           error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

static GValueArray *
mhs_lms_convert_login_info_to_garray (const MhsLoginInfo *old_info)
{
  GValueArray *value_array;
  int i;
  GValue empty_string_value = { 0 };
  guint null_fields = 0;
  GValue null_fields_value = { 0 };

  g_value_init (&empty_string_value, G_TYPE_STRING);

  value_array = g_value_array_new (sizeof (MhsLoginInfo) / sizeof (gchar *));

  for (i = 0; i < sizeof (MhsLoginInfo) / sizeof (gchar *); i++)
    {
      GValue *value;

      g_value_array_append (value_array, &empty_string_value);
      if (((gchar **) old_info)[i])
        {
          value = g_value_array_get_nth (value_array, i);
          g_value_set_string (value, ((gchar **) old_info)[i]);
        }
      else
        null_fields |= 1 << i;
    }

  g_value_unset (&empty_string_value);

  g_value_init (&null_fields_value, G_TYPE_UINT);
  g_value_set_uint (&null_fields_value, null_fields);
  g_value_array_append (value_array, &null_fields_value);

  return value_array;
}

gboolean
mhs_lms_modify_login (MhsLoginManagerStorage *self,
                      const MhsLoginInfo *old_info,
                      GHashTable *new_values,
                      GError **error)
{
  MhsLoginManagerStoragePrivate *priv;
  gboolean ret;
  GValueArray *value_array;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_lms_check_proxy (self, error))
    return FALSE;

  value_array = mhs_lms_convert_login_info_to_garray (old_info);

  ret = org_moblin_mhs_LoginManagerStorage_modify_login (priv->proxy,
                                                         value_array,
                                                         new_values,
                                                         error);

  g_value_array_free (value_array);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_lms_remove_all_logins (MhsLoginManagerStorage *self,
                           GError **error)
{
  MhsLoginManagerStoragePrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_lms_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_LoginManagerStorage_remove_all_logins (priv->proxy,
                                                              error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_lms_remove_login (MhsLoginManagerStorage *self,
                      const MhsLoginInfo *old_info,
                      GError **error)
{
  MhsLoginManagerStoragePrivate *priv;
  gboolean ret;
  GValueArray *value_array;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_lms_check_proxy (self, error))
    return FALSE;

  value_array = mhs_lms_convert_login_info_to_garray (old_info);

  ret = org_moblin_mhs_LoginManagerStorage_remove_login (priv->proxy,
                                                         value_array,
                                                         error);

  g_value_array_free (value_array);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_lms_search_logins (MhsLoginManagerStorage *self,
                       GHashTable *match_data,
                       guint *n_logins,
                       MhsLoginInfo **logins,
                       GError **error)
{
  MhsLoginManagerStoragePrivate *priv;
  GPtrArray *login_array;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_lms_check_proxy (self, error))
    return FALSE;

  if (org_moblin_mhs_LoginManagerStorage_search_logins (priv->proxy,
                                                        match_data,
                                                        &login_array,
                                                        error))
    {
      mhs_lms_convert_login_array_to_login_infos (login_array,
                                                  n_logins, logins);

      return TRUE;
    }
  else
    {
      _mhs_error_translate_from_dbus (error);

      return FALSE;
    }
}

gboolean
mhs_lms_set_login_saving_enabled (MhsLoginManagerStorage *self,
                                  const gchar *host,
                                  gboolean is_enabled,
                                  GError **error)
{
  MhsLoginManagerStoragePrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_lms_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_LoginManagerStorage_set_login_saving_enabled
    (priv->proxy, host, is_enabled,
     error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

void
mhs_lms_free_login_infos (guint n_logins, MhsLoginInfo *login_infos)
{
  guint i;

  for (i = 0; i < n_logins; i++)
    {
      g_free (login_infos[i].hostname);
      g_free (login_infos[i].form_submit_url);
      g_free (login_infos[i].http_realm);
      g_free (login_infos[i].username);
      g_free (login_infos[i].password);
      g_free (login_infos[i].username_field);
      g_free (login_infos[i].password_field);
    }

  g_free (login_infos);
}
