/* mhs-cookies.c */

#include <nsICookieService.h>
#include <nsICookieManager.h>
#include <nsICookieManager2.h>
#include <nsICookie.h>
#include <nsICookie2.h>
#include <nsIFile.h>
#include <nsILocalFile.h>
#include <nsIObserver.h>
#include <nsIObserverService.h>
#include <nsISupportsPrimitives.h>
#include <nsCOMPtr.h>
#include <nsMemory.h>
#include <nsServiceManagerUtils.h>
#include <nsWeakReference.h>
#include <nsStringGlue.h>
#include <nsCRTGlue.h>
#include <nsNetUtil.h>

#include "mhs-cookies.h"
#include "mhs-error-private.h"
#include "mhs-service.h"
#include "mhs-marshal.h"

/* D-Bus method implementations */

static gboolean mhs_cookies_get_cookie_string (MhsCookies   *self,
                                               const gchar  *uri,
                                               gchar       **cookie,
                                               GError      **error);
static gboolean mhs_cookies_get_cookie_string_from_http (MhsCookies   *self,
                                                         const gchar  *uri,
                                                         const gchar  *first_uri,
                                                         gchar       **cookie,
                                                         GError      **error);
static gboolean mhs_cookies_set_cookie_string (MhsCookies   *self,
                                               const gchar  *uri,
                                               const gchar  *cookie,
                                               GError      **error);
static gboolean mhs_cookies_set_cookie_string_from_http (MhsCookies   *self,
                                                         const gchar  *uri,
                                                         const gchar  *first_uri,
                                                         const gchar  *cookie,
                                                         const gchar  *time,
                                                         GError      **error);
static gboolean mhs_cookies_remove_all (MhsCookies   *self,
                                        GError     **error);
static gboolean mhs_cookies_get_all (MhsCookies  *self,
                                     GPtrArray  **cookies, // GPtrArray{GValueArray}
                                     GError     **error);
static gboolean mhs_cookies_remove (MhsCookies   *self,
                                    const gchar  *domain,
                                    const gchar  *name,
                                    const gchar  *path,
                                    gboolean      blocked,
                                    GError      **error);
static gboolean mhs_cookies_add (MhsCookies   *self,
                                 const gchar  *domain,
                                 const gchar  *path,
                                 const gchar  *name,
                                 const gchar  *value,
                                 gboolean      is_secure,
                                 gboolean      is_http_only,
                                 gboolean      is_session,
                                 gint64        expiry,
                                 GError      **error);
static gboolean mhs_cookies_count_cookies_from_host (MhsCookies   *self,
                                                     const gchar  *host,
                                                     guint        *count,
                                                     GError      **error);
static gboolean mhs_cookies_import_cookies (MhsCookies   *self,
                                            const gchar  *file,
                                            GError      **error);

#include "mhs-cookies-glue.h"

/* End D-Bus method implementations */

G_DEFINE_TYPE (MhsCookies, mhs_cookies, G_TYPE_OBJECT)

#define COOKIES_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MHS_TYPE_COOKIES, MhsCookiesPrivate))

struct _MhsCookiesPrivate
{
  nsICookieService *cookie_service;
};

enum
{
  COOKIE_CHANGED_SIGNAL,
  COOKIE_REJECTED_SIGNAL,

  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0, };

#define RETURN_IF_NS_FAILED \
  if (NS_FAILED (rv)) \
    { \
      mhs_error_set_from_nsresult (rv, error); \
      return FALSE; \
    }

#define RETURN_IF_NO_SERVICE \
  if (!self->priv->cookie_service) \
    { \
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error); \
      return FALSE; \
    }

static void
mhs_cookies_finalize (GObject *object)
{
  MhsCookies *self = MHS_COOKIES (object);
  MhsCookiesPrivate *priv = self->priv;

  // Release reference on cookie service
  if (priv->cookie_service)
    NS_RELEASE (priv->cookie_service);

  G_OBJECT_CLASS (mhs_cookies_parent_class)->finalize (object);
}

static void
mhs_cookies_class_init (MhsCookiesClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MhsCookiesPrivate));

  object_class->finalize = mhs_cookies_finalize;

  signals[COOKIE_CHANGED_SIGNAL] =
    g_signal_new ("cookie-changed",
                  G_TYPE_FROM_CLASS (object_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsCookiesClass, cookie_changed),
                  NULL, NULL,
                  mhs_marshal_VOID__STRING_STRING,
                  G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_STRING);

  signals[COOKIE_REJECTED_SIGNAL] =
    g_signal_new ("cookie-rejected",
                  G_TYPE_FROM_CLASS (object_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsCookiesClass, cookie_rejected),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1, G_TYPE_STRING);

  dbus_g_object_type_install_info (MHS_TYPE_COOKIES,
                                   &dbus_glib_mhs_cookies_object_info);
}

static void
mhs_cookies_init (MhsCookies *self)
{
  nsresult rv;
  DBusGConnection *connection;

  GError *error = NULL;
  MhsCookiesPrivate *priv = self->priv = COOKIES_PRIVATE (self);

  nsCOMPtr<nsICookieService> cookie_service =
    do_GetService (NS_COOKIESERVICE_CONTRACTID, &rv);
  if (NS_SUCCEEDED (rv))
    {
      priv->cookie_service = cookie_service;
      NS_ADDREF (priv->cookie_service);
    }
  else
    g_warning ("Failed to retrieve cookies service object");

  if ((connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
    }
  else
    {
      dbus_g_connection_register_g_object (connection,
                                           MHS_SERVICE_COOKIES_PATH,
                                           G_OBJECT (self));
      dbus_g_connection_unref (connection);
    }
}

MhsCookies *
mhs_cookies_new (void)
{
  return (MhsCookies *)g_object_new (MHS_TYPE_COOKIES, NULL);
}

/* nsICookieService bindings */
static gboolean
mhs_cookies_get_cookie_string (MhsCookies   *self,
                               const gchar  *uri,
                               gchar       **cookie,
                               GError      **error)
{
  nsresult rv;
  MhsCookiesPrivate *priv = self->priv;

  RETURN_IF_NO_SERVICE

  nsCOMPtr<nsIURI> nsuri;
  rv = NS_NewURI (getter_AddRefs (nsuri), uri);

  RETURN_IF_NS_FAILED

  rv = priv->cookie_service->GetCookieString (nsuri, nsnull, cookie);

  RETURN_IF_NS_FAILED

  return TRUE;
}

static gboolean
mhs_cookies_get_cookie_string_from_http (MhsCookies   *self,
                                         const gchar  *uri,
                                         const gchar  *first_uri,
                                         gchar       **cookie,
                                         GError      **error)
{
  nsresult rv;
  MhsCookiesPrivate *priv = self->priv;

  RETURN_IF_NO_SERVICE

  nsCOMPtr<nsIURI> nsuri;
  rv = NS_NewURI (getter_AddRefs (nsuri), uri);

  RETURN_IF_NS_FAILED

  nsCOMPtr<nsIURI> first_nsuri;
  rv = NS_NewURI (getter_AddRefs (first_nsuri), uri);

  RETURN_IF_NS_FAILED

  rv = priv->cookie_service->
    GetCookieStringFromHttp (nsuri, first_nsuri, nsnull, cookie);

  RETURN_IF_NS_FAILED

  return TRUE;
}

static gboolean
mhs_cookies_set_cookie_string (MhsCookies   *self,
                               const gchar  *uri,
                               const gchar  *cookie,
                               GError      **error)
{
  nsresult rv;
  MhsCookiesPrivate *priv = self->priv;

  RETURN_IF_NO_SERVICE

  nsCOMPtr<nsIURI> nsuri;
  rv = NS_NewURI (getter_AddRefs (nsuri), uri);

  RETURN_IF_NS_FAILED

  rv = priv->cookie_service->SetCookieString (nsuri, nsnull, cookie, nsnull);

  RETURN_IF_NS_FAILED

  return TRUE;
}

static gboolean
mhs_cookies_set_cookie_string_from_http (MhsCookies   *self,
                                         const gchar  *uri,
                                         const gchar  *first_uri,
                                         const gchar  *cookie,
                                         const gchar  *time,
                                         GError      **error)
{
  nsresult rv;
  MhsCookiesPrivate *priv = self->priv;

  RETURN_IF_NO_SERVICE

  nsCOMPtr<nsIURI> nsuri;
  rv = NS_NewURI (getter_AddRefs (nsuri), uri);

  RETURN_IF_NS_FAILED

  nsCOMPtr<nsIURI> first_nsuri;
  rv = NS_NewURI (getter_AddRefs (first_nsuri), uri);

  RETURN_IF_NS_FAILED

  rv = priv->cookie_service->
    SetCookieStringFromHttp (nsuri, first_nsuri, nsnull, cookie, time, nsnull);

  RETURN_IF_NS_FAILED

  return TRUE;
}

/* nsICookieManager bindings */
static gboolean
mhs_cookies_remove_all (MhsCookies   *self,
                        GError     **error)
{
  nsresult rv;
  MhsCookiesPrivate *priv = self->priv;

  RETURN_IF_NO_SERVICE

  nsCOMPtr<nsICookieManager> cookie_manager;
  cookie_manager = do_QueryInterface (priv->cookie_service, &rv);

  RETURN_IF_NS_FAILED

  rv = cookie_manager->RemoveAll ();

  RETURN_IF_NS_FAILED

  return TRUE;
}

static gboolean
mhs_cookies_get_all (MhsCookies  *self,
                     GPtrArray  **cookies, // GPtrArray{GValueArray}
                     GError     **error)
{
  nsresult rv;
  MhsCookiesPrivate *priv = self->priv;

  RETURN_IF_NO_SERVICE

  nsCOMPtr<nsICookieManager> cookie_manager;
  cookie_manager = do_QueryInterface (priv->cookie_service, &rv);

  RETURN_IF_NS_FAILED

  nsISimpleEnumerator *enumerator;

  rv = cookie_manager->GetEnumerator (&enumerator);

  RETURN_IF_NS_FAILED

  *cookies = g_ptr_array_new ();

#define BREAK_IF_NS_FAILED \
  if (NS_FAILED (rv)) \
    { \
      for (guint i = 0; i < (*cookies)->len; i++) \
        g_value_array_free ((GValueArray *)((*cookies)->pdata[i])); \
      g_ptr_array_free ((*cookies), TRUE); \
      *cookies = NULL; \
      break; \
    }

  PRBool has_more;
  while (NS_SUCCEEDED (rv = enumerator->HasMoreElements (&has_more)) &&
         has_more)
    {
      nsISupports *element;
      rv = enumerator->GetNext (&element);

      BREAK_IF_NS_FAILED

      nsICookie *cookie = static_cast<nsICookie *>(element);
      nsICookie2 *cookie2 = static_cast<nsICookie2 *>(element);

      nsCAutoString name, value, host, path;
      rv = cookie->GetName (name);
      BREAK_IF_NS_FAILED
      rv = cookie->GetValue (value);
      BREAK_IF_NS_FAILED
      rv = cookie->GetHost (host);
      BREAK_IF_NS_FAILED
      rv = cookie->GetPath (path);
      BREAK_IF_NS_FAILED

      PRInt64 expiry, creation_time;
      rv = cookie2->GetExpiry (&expiry);
      BREAK_IF_NS_FAILED
      rv = cookie2->GetCreationTime (&creation_time);
      BREAK_IF_NS_FAILED

      PRBool is_session, is_secure, is_http_only;
      rv = cookie2->GetIsSession (&is_session);
      BREAK_IF_NS_FAILED
      rv = cookie->GetIsSecure (&is_secure);
      BREAK_IF_NS_FAILED
      rv = cookie2->GetIsHttpOnly (&is_http_only);
      BREAK_IF_NS_FAILED

      // NOTE: If values are added or removed to this array, don't forget
      //       to alter the number below.
      GValueArray *array = g_value_array_new (9);
      GValue gvalue = { 0, };

      g_value_init (&gvalue, G_TYPE_STRING);
      g_value_set_static_string (&gvalue, name.get());
      g_value_array_append (array, &gvalue);
      g_value_set_static_string (&gvalue, value.get());
      g_value_array_append (array, &gvalue);
      g_value_set_static_string (&gvalue, host.get());
      g_value_array_append (array, &gvalue);
      g_value_set_static_string (&gvalue, path.get());
      g_value_array_append (array, &gvalue);

      gvalue = (GValue){ 0, };
      g_value_init (&gvalue, G_TYPE_INT64);
      g_value_set_int64 (&gvalue, expiry);
      g_value_array_append (array, &gvalue);
      g_value_set_int64 (&gvalue, creation_time);
      g_value_array_append (array, &gvalue);

      gvalue = (GValue){ 0, };
      g_value_init (&gvalue, G_TYPE_BOOLEAN);
      g_value_set_boolean (&gvalue, is_session);
      g_value_array_append (array, &gvalue);
      g_value_set_boolean (&gvalue, is_secure);
      g_value_array_append (array, &gvalue);
      g_value_set_boolean (&gvalue, is_http_only);
      g_value_array_append (array, &gvalue);

      g_ptr_array_add (*cookies, (gpointer)array);
    }

#undef BREAK_IF_NS_FAILED

  RETURN_IF_NS_FAILED

  return TRUE;
}

static gboolean
mhs_cookies_remove (MhsCookies   *self,
                    const gchar  *domain,
                    const gchar  *name,
                    const gchar  *path,
                    gboolean      blocked,
                    GError      **error)
{
  nsresult rv;
  MhsCookiesPrivate *priv = self->priv;

  RETURN_IF_NO_SERVICE

  nsCOMPtr<nsICookieManager> cookie_manager;
  cookie_manager = do_QueryInterface (priv->cookie_service, &rv);

  RETURN_IF_NS_FAILED

  rv = cookie_manager->Remove (nsCAutoString (domain),
                               nsCAutoString (name),
                               nsCAutoString (path),
                               blocked);

  RETURN_IF_NS_FAILED

  return TRUE;
}

/* nsICookieManager2 bindings */
static gboolean
mhs_cookies_add (MhsCookies   *self,
                 const gchar  *domain,
                 const gchar  *path,
                 const gchar  *name,
                 const gchar  *value,
                 gboolean      is_secure,
                 gboolean      is_http_only,
                 gboolean      is_session,
                 gint64        expiry,
                 GError      **error)
{
  nsresult rv;
  MhsCookiesPrivate *priv = self->priv;

  RETURN_IF_NO_SERVICE

  nsCOMPtr<nsICookieManager2> cookie_manager2;
  cookie_manager2 = do_QueryInterface (priv->cookie_service, &rv);

  RETURN_IF_NS_FAILED

  rv = cookie_manager2->Add (nsCAutoString (domain),
                             nsCAutoString (name),
                             nsCAutoString (path),
                             nsCAutoString (name),
                             is_secure,
                             is_http_only,
                             is_session,
                             expiry);

  RETURN_IF_NS_FAILED

  return TRUE;
}

static gboolean
mhs_cookies_count_cookies_from_host (MhsCookies   *self,
                                     const gchar  *host,
                                     guint        *count,
                                     GError      **error)
{
  nsresult rv;
  MhsCookiesPrivate *priv = self->priv;

  RETURN_IF_NO_SERVICE

  nsCOMPtr<nsICookieManager2> cookie_manager2;
  cookie_manager2 = do_QueryInterface (priv->cookie_service, &rv);

  RETURN_IF_NS_FAILED

  rv = cookie_manager2->CountCookiesFromHost (nsCAutoString (host),
                                              count);

  RETURN_IF_NS_FAILED

  return TRUE;
}

static gboolean
mhs_cookies_import_cookies (MhsCookies   *self,
                            const gchar  *path,
                            GError      **error)
{
  nsresult rv;
  MhsCookiesPrivate *priv = self->priv;

  RETURN_IF_NO_SERVICE

  nsCOMPtr<nsICookieManager2> cookie_manager2;
  cookie_manager2 = do_QueryInterface (priv->cookie_service, &rv);

  RETURN_IF_NS_FAILED

  nsCOMPtr<nsILocalFile> local_file =
    do_GetService ("@mozilla.org/file/local;1", &rv);

  RETURN_IF_NS_FAILED

  rv = local_file->InitWithPath (NS_ConvertUTF8toUTF16 (path));

  RETURN_IF_NS_FAILED

  nsCOMPtr<nsIFile> file = do_QueryInterface (local_file, &rv);

  RETURN_IF_NS_FAILED

  rv = cookie_manager2->ImportCookies (file);

  RETURN_IF_NS_FAILED

  return TRUE;
}

